! granular temperature equation solver
! author: Stefanie Rynders

      module ice_gtemp

      use ice_kinds_mod
      use ice_dyn_shared ! everything
      use ice_constants, only: pi, p5, p166,c1
      use ice_domain_size, only: max_blocks, nx_global, ny_global
      use ice_blocks, only: block, nx_block, ny_block
      use ice_state, only: trcrn, nt_gtemp

      implicit none
      private
      public :: init_gtemp, step_gtemp, write_restart_gtemp, read_restart_gtemp
      save

      real (kind=dbl_kind), dimension (nx_block,ny_block,max_blocks), public  :: &
         gtemp           , &  ! granular temperature 
         gt_stresscontr  , &
         gt_diffusion    , &
         gt_tursource    , &
         gt_rubsink      , &
         gt_collisionsink, &
         gt_waves        , &
         gt_total

      real (kind=dbl_kind), dimension (nx_block,ny_block,max_blocks), public  :: &
         wave_amp        , &
         wave_per        , &
         wave_length     , &
         ocean_depth

      ! namelist parameters

      logical (kind=log_kind), public :: &
         restart_gtemp         ! if .true., read gtemp restart file

!=======================================================================

      contains

!=======================================================================

! Initialize parameters and variables needed for the evcp dynamics
! (based on init_eap)

      subroutine init_gtemp (nx_block, ny_block, ncat, trgtemp)

      use ice_constants, only: c0

      integer(kind=int_kind), intent(in) :: &
             nx_block , &
             ny_block , &
             ncat

      real(kind=dbl_kind), dimension(nx_block,ny_block,ncat), &
             intent(out) :: &
             trgtemp  ! granular temperature

      trgtemp(:,:,:) = c0
      gtemp(:,:,:) = c0

      end subroutine init_gtemp

!=======================================================================
! calculation of new granular temperature

      subroutine step_gtemp  (nx_block,    ny_block,  &
                              max_blocks,  icellt,    &
                              indxti,      indxtj,    &
                              uvel,        vvel,      &
                              dxt,         dyt,       &
                              gtemp,       aice,      &
                              vice,        vsno,      &
                              dfloe,       maxfloe,   &
                              Lf,          tmass,     &
                              dt,          strength,  &
                              stressp_1,   stressp_2, &
                              stressp_3,   stressp_4, &
                              stressm_1,   stressm_2, &
                              stressm_3,   stressm_4, &
                              stress12_1,  stress12_2,&
                              stress12_3,  stress12_4,&
                              divu,        tarear,    &
                              tension,     shear,     &     
                              gt_stresscontr , &
                              gt_diffusion   , &
                              gt_tursource   , &
                              gt_rubsink     , &
                              gt_collisionsink, &
                              gt_waves       , &
                              wave_amp        , &
                              wave_per        , &
                              wave_length     , &
                              ocean_depth     , &
                              gt_total, iblk, &
                              this_block)

      use ice_zbgc_shared, only: rhosi
      use ice_calendar, only: time
      use ice_constants, only: c0, c1, c1p5, c2, p01, p25, p5, c20, p1, p333, &
                               rhoi, rhos, puny, field_loc_center, field_type_scalar, &
                               pi
      use ice_domain_size, only: ncat
      use ice_dyn_evcp, only: resc
      use ice_dyn_shared, only: Pstar, Cstar
      use ice_floe, only: Astar, set_floesize, Lf_const
      USE iom ! I/O manager library

      integer (kind=int_kind), intent(in) :: &
         nx_block, ny_block, & ! block dimensions
         max_blocks        , &
         icellt            , &
         iblk

      integer (kind=int_kind), dimension (nx_block*ny_block), intent(in) :: &
         indxti   , & ! compressed index in i-direction
         indxtj       ! compressed index in j-direction

      real (kind=dbl_kind), intent(in) :: &
         dt      ! time step

      real (kind=dbl_kind), dimension (nx_block,ny_block), intent(in) :: &
         strength , & ! ice strength (N/m)
         uvel     , & ! x-component of velocity (m/s)
         vvel     , & ! y-component of velocity (m/s)
         dxt      , & ! width of T-cell through the middle (m)
         dyt      , & ! height of T-cell through the middle (m)
         vice     , & ! volume per unit area of ice (m)
         vsno     , & ! volume per unit area of snow (m)
         aice     , & ! concentration of ice
         tmass    , & ! total mass of ice and snow (kg/m^2)
         divu     , &
         tension  , &
         shear    , &
         tarear   , &
         maxfloe  , &
         wave_amp , &
         wave_per , &
         wave_length , &
         ocean_depth

      real (kind=dbl_kind), dimension (nx_block,ny_block), intent(inout) :: &
         Lf       , &
         dfloe        ! distance s between floes

      real (kind=dbl_kind), dimension (nx_block,ny_block), &
         intent(inout) :: &
         stressp_1, stressp_2, stressp_3, stressp_4 , & ! sigma11+sigma22
         stressm_1, stressm_2, stressm_3, stressm_4 , & ! sigma11-sigma22
         stress12_1,stress12_2,stress12_3,stress12_4    ! sigma12     

      real (kind=dbl_kind), dimension (nx_block,ny_block), intent(inout) :: &
         gtemp          , &  ! granular temperature
         gt_stresscontr , &
         gt_diffusion   , &
         gt_tursource   , &
         gt_rubsink     , &
         gt_collisionsink, &
         gt_waves, &
         gt_total

      type (block) :: &
         this_block           ! block information for current block

      ! local variables

      integer (kind=int_kind) :: &
         i, j, ij, n, helpi, helpj

      integer (kind=int_kind), parameter :: &
         lambdac = c1  , &  ! dimensionless number diffusion
         Faccent = c2       ! strength of fluctuations coupling (kg/(m^2s^2))

      real (kind=dbl_kind) :: &
         helpvariable

      real (kind=dbl_kind), dimension (nx_block,ny_block) :: &
        tmassdt , & ! mass of T-cell/dt (kg/m^2/s) 
        tmassdti , & ! invers mass of T-cell/dt (m^2 s/kg) 
        gip, gim, gjp, gjm, &
        sip, sim, sjp, sjm, &
        Lfip, Lfim, Lfjp, Lfjm, &
        dxp, dyp, dxm, dym, &
        stress11 , &
        stress22 , &
        stress12 , &
        stressp  , &
        stressm  , &
        dudx,dvdy, &
        gtempn   , &
        gP

      !-----------------------------------------------------------------
      ! Initialize
      !-----------------------------------------------------------------          
        do n=1, ncat
        do j = 1, ny_block
        do i = 1, nx_block
           gtemp(i,j)=trcrn(i,j,nt_gtemp,n,iblk)
        enddo
        enddo
        enddo

        do j = 1, ny_block
        do i = 1, nx_block
         if (aice(i,j).le. puny) then
            gtemp(i,j) = c0 
         endif
        enddo
        enddo

         do j = 1, ny_block
            do i = 1, nx_block
               gtempn(i,j) = c0
            enddo
         enddo

      !-----------------------------------------------------------------
      ! Define variables
      !-----------------------------------------------------------------          

        do ij = 1, icellt
         i = indxti(ij)
         j = indxtj(ij)

         tmassdt(i,j) = tmass(i,j)/dt ! kg/m^2/s
         tmassdti(i,j) = dt/tmass(i,j) ! m^2 s/kg


! have to update the floesize if aice has changed when using profile
         if ((Lf(i,j).eq.c0) .or. (dfloe(i,j).eq.c0) .or. (Lf_const.eq.2) .or. (Lf_const.eq.5)) then 
           call set_floesize (Lf      , dfloe , &
                              maxfloe , aice  , &
                              i       , j       )
         endif

! FOR rubsink
         gP(i,j) =  strength(i,j)/vice(i,j)   ! N/m2

! FOR DIVERGENCE
         ! define distances
         dxp(i,j) = 0.5*dxt(i,j) + 0.5*dxt(i+1,j)   ! m
         dxm(i,j) = 0.5*dxt(i,j) + 0.5*dxt(i-1,j)

         dyp(i,j) = 0.5*dyt(i,j) + 0.5*dyt(i,j+1)
         dym(i,j) = 0.5*dyt(i,j) + 0.5*dyt(i,j-1)

         ! define half index gtemp values    (improve weighted average?)
         gip(i,j) = p5*(gtemp(i,j)+gtemp(i+1,j))   ! m2/s2
         gim(i,j) = p5*(gtemp(i,j)+gtemp(i-1,j))
         gjp(i,j) = p5*(gtemp(i,j)+gtemp(i,j+1))
         gjm(i,j) = p5*(gtemp(i,j)+gtemp(i,j-1))

         ! define half index s values   (improve weighted average?)
         sip(i,j) = p5*(dfloe(i,j)+dfloe(i+1,j))   ! m
! 20/07/16 plus sign wrong, should be minus
         sim(i,j) = p5*(dfloe(i,j)+dfloe(i-1,j))
         sjp(i,j) = p5*(dfloe(i,j)+dfloe(i,j+1))
         sjm(i,j) = p5*(dfloe(i,j)+dfloe(i,j-1))

         ! define half index Lf values   (improve weighted average?)
         Lfip(i,j) = p5*(Lf(i,j)+Lf(i+1,j))   ! m
         Lfim(i,j) = p5*(Lf(i,j)+Lf(i-1,j))
         Lfjp(i,j) = p5*(Lf(i,j)+Lf(i,j+1))
         Lfjm(i,j) = p5*(Lf(i,j)+Lf(i,j-1))

!FOR STRESSES
! nr 1 to 4 are defined in the corners of the T cell, take the average
! to get T point value
         stressp(i,j) = p25*(stressp_1(i,j) + stressp_2(i,j) &
                        + stressp_3(i,j) + stressp_4(i,j))
         stressm(i,j) = p25*(stressm_1(i,j) + stressm_2(i,j) &
                        + stressm_3(i,j) + stressm_4(i,j))
         stress11(i,j) = p5*(stressp(i,j) + stressm(i,j))
         stress22(i,j) = p5*(stressp(i,j) - stressm(i,j))
         stress12(i,j) = p25*(stress12_1(i,j) + &
                         stress12_2(i,j) + stress12_3(i,j) + stress12_4(i,j))

         ! divide by the thickness to get stresses in N/m2
         stress11(i,j) = stress11(i,j)/vice(i,j)
         stress22(i,j) = stress22(i,j)/vice(i,j)
         stress12(i,j) = stress12(i,j)/vice(i,j)

         ! gradients of velocities from strain rates
         ! divu, tension, shear are T point values
         dudx(i,j) = p5*(divu(i,j) + tension(i,j))  ! (1/s)
         dvdy(i,j) = p5*(divu(i,j) - tension(i,j))

      enddo !ij

      !-----------------------------------------------------------------
      ! Compute sources/sinks
      !-----------------------------------------------------------------  

      do ij = 1, icellt
         i = indxti(ij)
         j = indxtj(ij)

       gt_diffusion(i,j)=lambdac*rhosi* &
       (Lfip(i,j)*Lfip(i,j)*sqrt(gip(i,j))/sip(i,j)*(gtemp(i+1,j)-gtemp(i,j))/(dxp(i,j)*dxt(i,j))&
        -Lfim(i,j)*Lfim(i,j)*sqrt(gim(i,j))/sim(i,j)*(gtemp(i,j)-gtemp(i-1,j))/(dxm(i,j)*dxt(i,j))&
        +Lfjp(i,j)*Lfjp(i,j)*sqrt(gjp(i,j))/sjp(i,j)*(gtemp(i,j+1)-gtemp(i,j))/(dyp(i,j)*dyt(i,j))&
        -Lfjm(i,j)*Lfjm(i,j)*sqrt(gjm(i,j))/sjm(i,j)*(gtemp(i,j)-gtemp(i,j-1))/(dym(i,j)*dyt(i,j)))

       gt_stresscontr(i,j)=-1*stress11(i,j)*dudx(i,j)-stress12(i,j)*shear(i,j)*p5 &
                           -stress22(i,j)*dvdy(i,j) 
       gt_tursource(i,j) = aice(i,j)*Faccent*sqrt(gtemp(i,j))
       gt_rubsink(i,j) = -p25*gP(i,j)/Lf(i,j)*sqrt(gtemp(i,j))
       gt_collisionsink(i,j) = - resc*rhosi/dfloe(i,j)*gtemp(i,j)**c1p5

! add energy input from waves
       if ((aice(i,j).gt. 0.01) .and. (wave_per(i,j).gt.0.01) .and.( wave_length(i,j).gt. 0.01)) then
         if ((wave_per(i,j).lt. 100000.) .and.(wave_length(i,j).lt.100000.) .and.( wave_amp(i,j).lt.100000.)) then
         gt_waves(i,j) = 4*wave_amp(i,j)**2/wave_per(i,j)**2/ &
                     tanh(2*pi/wave_length(i,j)*ocean_depth(i,j))**2
         else 
           gt_waves(i,j) = c0
         endif
       else
         gt_waves(i,j) = c0
       endif

       if (aice(i,j).le. puny) then
       gt_stresscontr(i,j)= c0
       gt_rubsink(i,j) = c0
       endif

        gt_total(i,j)=gt_diffusion(i,j)+gt_stresscontr(i,j)+gt_tursource(i,j) &
                      +gt_rubsink(i,j)+gt_collisionsink(i,j)+gt_waves(i,j)

      enddo !ij

      !-----------------------------------------------------------------
      ! change units to m2/s2
      !-----------------------------------------------------------------  

      do ij = 1, icellt
         i = indxti(ij)
         j = indxtj(ij)
         gt_diffusion(i,j) = gt_diffusion(i,j)*vice(i,j)*tmassdti(i,j) 
         gt_stresscontr(i,j) = gt_stresscontr(i,j)*vice(i,j)*tmassdti(i,j) 
         gt_tursource(i,j) = gt_tursource(i,j)*vice(i,j)*tmassdti(i,j) 
         gt_rubsink(i,j) = gt_rubsink(i,j)*vice(i,j)*tmassdti(i,j)  
         gt_collisionsink(i,j) = gt_collisionsink(i,j)*vice(i,j)*tmassdti(i,j)
         gt_waves(i,j) = gt_waves(i,j)*vice(i,j)*tmassdti(i,j)
         gt_total(i,j) = gt_total(i,j)*vice(i,j)*tmassdti(i,j)

         if (aice(i,j).le. puny) then
          gt_diffusion(i,j) = c0 
           gt_tursource(i,j) = c0 
           gt_collisionsink(i,j) = c0
           gt_total(i,j) = c0
           gt_stresscontr(i,j)= c0
           gt_rubsink(i,j) = c0
         endif

         if (aice(i,j).le. 0.01) then
           gt_waves(i,j) = c0
         endif

      enddo !ij

      !-----------------------------------------------------------------
      ! Compute new granular temperature
      !-----------------------------------------------------------------  

      do ij = 1, icellt
         i = indxti(ij)
         j = indxtj(ij)
         gtempn(i,j) = gtemp(i,j) &
         + gt_diffusion(i,j)      &
         + gt_stresscontr(i,j)    &
         + gt_tursource(i,j)      &
         + gt_rubsink(i,j)        &
         + gt_collisionsink(i,j)  &
         + gt_waves(i,j)

      enddo !ij

      !-----------------------------------------------------------------
      ! update tracer
      !-----------------------------------------------------------------  

      do ij = 1, icellt
         i = indxti(ij)
         j = indxtj(ij)
         gtemp(i,j) = gtempn(i,j)
         if (aice(i,j).le. puny) then
            gtemp(i,j) = c0
            gtempn(i,j) = c0
         endif
      enddo !ij


!cap at zero
      do ij = 1, icellt
         i = indxti(ij)
         j = indxtj(ij)
         if (gtempn(i,j).lt. puny) then
            gtemp(i,j) = c0     
         endif
      enddo !ij

         do n = 1, ncat
         do j = 1, ny_block
         do i = 1, nx_block
            if (aice(i,j).eq. c0) then
               Lf(i,j) = c0 
            endif
         enddo
         enddo
         enddo
      end subroutine step_gtemp

!=======================================================================
      subroutine write_restart_gtemp ()

      use ice_communicate, only: my_task, master_task
      use ice_domain_size, only: ncat
      use ice_fileunits, only: nu_diag, nu_dump_gtemp
      use ice_state, only: trcrn, nt_gtemp
      use ice_restart,only: write_restart_field

      ! local variables

      logical (kind=log_kind) :: diag

      diag = .true.

      !-----------------------------------------------------------------

      call write_restart_field(nu_dump_gtemp,0,trcrn(:,:,nt_gtemp,:,:), &
                               'ruf8','gtemp',ncat,diag)

      end subroutine write_restart_gtemp

!=======================================================================

!  restart

      subroutine read_restart_gtemp()

      use ice_communicate, only: my_task, master_task
      use ice_domain_size, only: ncat
      use ice_fileunits, only: nu_diag, nu_restart_gtemp
      use ice_state, only: trcrn, nt_gtemp
      use ice_restart,only: read_restart_field

      ! local variables

      logical (kind=log_kind) :: &
         diag

      diag = .true.

      if (my_task == master_task) write(nu_diag,*) &
         'granular temperature (m2/s2)'

      call read_restart_field(nu_restart_gtemp,0,trcrn(:,:,nt_gtemp,:,:), &
                              'ruf8','gtemp',ncat,diag)
      end subroutine read_restart_gtemp

!=======================================================================
      end module ice_gtemp

!=======================================================================

